/*****************************************************************************
 $Id: StegData.java,v 1.27 2002/04/28 23:03:36 bastian Exp $
 Part of TRex, (c) 2001, 2002 Bastian Friedrich <bastian@bastian-friedrich.de>
 This software licensed under the GPL.
 *****************************************************************************/
package trex;

import java.awt.*;
import java.awt.image.*;
import java.lang.*;
import javax.swing.*;
import java.util.*;
import javax.imageio.*;
import javax.imageio.stream.*;
import java.io.*;

import trex.Algo.*;
import trex.Filter.*;
import trex.GUI.*;

/**
 * This class handles steganography algorithms and their data.
 * @author Bastian Friedrich <a href="mailto:bastian@bastian-friedrich.de">&lt;bastian@bastian-friedrich.de&gt;</a>
 * @version $Revision: 1.27 $
 */

public class StegData {

  /** Data to encrypt */
  String containedData;
  /** ImageIcon containing the uncrypted image */
  ImageIcon uncryptedImg;
  /** ImageIcon containing the image with hidden data */
  ImageIcon cryptedImg;
  /** Used algorithm */
  StegoAlgo algo;
  /** parent window */
  MainFrame parent;

  /**
   * Default constructor.
   * Initializes member variables and algorithm.
   * @param parent Parent window for message boxes.
   * @param algoClass algorithm to use initially
   */
  public StegData(MainFrame parent, Class algoClass) {
    uncryptedImg = null;
    cryptedImg = null;
    containedData = null;
    this.parent = parent;
    boolean failure = false;

    try {
      algo = (StegoAlgo)algoClass.newInstance();
    }

    catch (Exception e) {
      failure = true;
    }

    if ((algo == null) || (!(algo instanceof trex.Algo.StegoAlgo)))
      failure = true;

    if (failure) {
      JOptionPane.showMessageDialog(parent,
                      "Initial algorithm could not be loaded! Desperately exiting.",
                      "Error",
                      JOptionPane.ERROR_MESSAGE);
      System.exit(-1);
    }
  }

  /**
   * Set a new algorithm.
   * @param algoClass New algorithm.
   * @return true on success; false on failure.
   */
  public boolean setAlgo(Class algoClass) {
    StegoAlgo newAlgo = null;
    boolean failure = false;

    try {
      newAlgo = (StegoAlgo)(algoClass.newInstance());
    }

    catch (Exception e) {
      failure = true;
      e.printStackTrace();
    }

    finally {

      if ((newAlgo == null) || (!(newAlgo instanceof trex.Algo.StegoAlgo))) {
        failure = true;
      }

      if (failure) {
        JOptionPane.showMessageDialog(parent,
                        "Algorithm could not be loaded! Switching back.",
                        "Error",
                        JOptionPane.ERROR_MESSAGE);
        return false;
      }

      algo = newAlgo;
      return true;
    }
  }

  /**
   * Get the uncrypted image.
   * @return {@link javax.swing.ImageIcon} containing the uncrypted Image.
   */
  public ImageIcon getUncrypted() {
    return uncryptedImg;
  }

  /**
   * Get the encrypted image.
   * @return {@link javax.swing.ImageIcon} containing the crypted Image.
   */
  public ImageIcon getCrypted() {
    return cryptedImg;
  }

  /**
   * Try to start the crypting algorithm. If this is impossible for any reason,
   * a {@link javax.swing.JOptionPane} is displayed.
   * @return Success.
   */
  private boolean tryCrypt() {
    if ((uncryptedImg == null)
     || (containedData == null)
     || (algo == null)
     || (!algo.validConfig())) {
      cryptedImg = null;
      return false;
    }

    if (!algo.pictureLargeEnough(uncryptedImg, containedData)) {
      JOptionPane.showMessageDialog(parent,
                                    "Picture too small for these data!",
                                    "Error",
                                    JOptionPane.ERROR_MESSAGE);
      return false;
    }

    cryptedImg = algo.getEncrypted(containedData, uncryptedImg);
    return true;
  }

  /**
   * Try to start the decryption algorithm. If this is successful,
   * it's data is read into <code>this</code>'s data.
   * @return Success.
   */
  private boolean tryDecrypt() {
    if ((cryptedImg == null)
     || (!algo.validConfig())) return false;

    uncryptedImg = null;

    try {
      this.containedData = algo.getDecrypted(cryptedImg);
    }

    catch (DecryptImpossibleException ex) {
      JOptionPane.showMessageDialog(parent,
         "Decryption failed. The algorithm was not able to handle the input data.\n" +
         "The picture possibly was not created with this algorithm or it's " +
         "configuration or pass phrase is invalid.",
         "Decryption failed",
         JOptionPane.ERROR_MESSAGE);
      return false;
    }
    return true;
  }

  /**
   * This routine is called whenever the parameters have changed, e.g. new
   * images, new data, changes in pass phrase/config dialog, ...
   * It then starts en- or decryption, depending on the availability of data.
   * @return true on success, false on failure.
   */
  public boolean paramsChanged() {
    if ((containedData != null) && (uncryptedImg != null)) {
      cryptedImg = null;
    } else if (cryptedImg != null) {
      uncryptedImg = null;
    }

    if (!algo.validConfig()) {
      JOptionPane.showMessageDialog(parent,
                                    "Config is not yet valid!\n" +
                                    "Please check the config dialog.",
                                    "Error",
                                    JOptionPane.ERROR_MESSAGE);


      return false;
    }
    if ((containedData != null) && (uncryptedImg != null)) {
      tryCrypt();
    } else if (cryptedImg != null) {
      tryDecrypt();
    }

    return true;
  }

  /**
   * Open an uncrypted Image with <code>filename</code>.
   * @param filename File name.
   * @return Success.
   */
  public boolean openUncrypted(String filename) {
    ImageIcon newImage = new ImageIcon(filename);

    if (newImage != null) {
      uncryptedImg = newImage;
      cryptedImg = null;
      paramsChanged();
      return true;
    }
    return false;
  }

  /**
   * Open a crypted Image (containing hidden data).
   * @param filename File name.
   * @return Success.
   */
  public boolean openCrypted(String filename) {
    ImageIcon newImage = new ImageIcon(filename);

    if (newImage != null) {
      cryptedImg = newImage;
      uncryptedImg = null;
      paramsChanged();
      return true;
    }
    return false;
  }

  /**
   * Save crypted image to file.
   * @param filename File name.
   * @return Success.
   */
  public boolean saveCrypted(String filename) {

    String suffix = null;
    int p = filename.lastIndexOf('.');

    if (p > 0 &&  p < filename.length() - 1) {
      suffix = filename.substring(p+1).toLowerCase();
    } else {
      JOptionPane.showMessageDialog(parent,
                  "File needs an extension. \"png\" is preferred.",
                  "Error",
                  JOptionPane.ERROR_MESSAGE);
      return false;
    }

    // Get Image writer.
    Iterator i = ImageIO.getImageWritersBySuffix(suffix);
    ImageWriter imgWriter = null;
    if (i.hasNext())
      imgWriter = (ImageWriter)i.next();

    if (imgWriter == null) {
      JOptionPane.showMessageDialog(parent,
                      "An unknown error occured during writing the file.",
                      "Error writing file",
                      JOptionPane.ERROR_MESSAGE);
      return false;
    }

    // Write to file
    try {
      File f = new File(filename);
      ImageOutputStream ios = ImageIO.createImageOutputStream(f);
      imgWriter.setOutput(ios);

      imgWriter.write(TRexUtil.iitobi(cryptedImg, parent));
    }
    // IO Error handling
    catch (Exception e) {
      JOptionPane.showMessageDialog(parent,
                      "An unknown error occured during writing the file.",
                      "Error writing file",
                      JOptionPane.ERROR_MESSAGE);
      return false;
    }

    return true;
  }

  /**
   * Set new data to hide.
   * @param data New text.
   */
  public void setData(String data) {
    containedData = data;
    paramsChanged();
    parent.resetPicPanels();
  }

  /**
   * Get contained data (e.g. from decryption).
   * @return Current text.
   */
  public String getData() {
    return containedData;
  }

  /**
   * Set a new pass phrase for current algorithm.
   * @param pp new pass phrase.
   * @return true on success, false on failure.
   */
  public boolean setPassPhrase(String pp) {
    if ((algo != null) && (algo.hasPassPhrase())) {
      algo.setPassPhrase(pp);

      paramsChanged();
      parent.resetPicPanels();
      parent.resetDataPanel();

      return true;
    } else return false;
  }

  /**
   * Return whether current algorithm needs a pass phrase.
   */
  public boolean hasPassPhrase() {
    if (algo != null)
      return algo.hasPassPhrase();
    else
      return false;
  }

  /**
   * Return whether current algorith has a config dialog.
   */
  public boolean hasConfigDialog() {
    if (algo != null)
      return algo.hasConfigDialog();
    else
      return false;
  }

  /**
   * Get current algorithm's config dialog.
   */
  public Component getConfigDialog() {
    if (algo != null)
      return algo.getConfigDialog();
    else
      return null;
  }

  /**
   * Get current algorithm's default amplification for combined panel.
   */
  public int defaultAmplification() {
    if (algo != null)
      return algo.defaultAmplification();
    else
      return 0;
  }

}
