/*****************************************************************************
 $Id: MainFrame.java,v 1.31 2002/04/28 21:37:26 bastian Exp $
 Part of TRex, (c) 2001, 2002 Bastian Friedrich <bastian@bastian-friedrich.de>
 This software licensed under the GPL.
 *****************************************************************************/
package trex.GUI;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.renderable.*;
import java.io.*;
import javax.swing.*;

import trex.GUI.ImageChooser.*;
import trex.*;
import trex.Algo.*;

/**
 * This class represents the main frame.
 * It handels menu, toolbar and tabbed pane.
 * @author Bastian Friedrich <a href="mailto:bastian@bastian-friedrich.de">&lt;bastian@bastian-friedrich.de&gt;</a>
 * @version $Revision: 1.31 $
 */

public class MainFrame extends JFrame {
  /** GUI elements */
  JPanel contentPane;
  JMenuBar jMenuBar1 = new JMenuBar();
  JMenu jMenuFile = new JMenu();
  JMenuItem jMenuFileExit = new JMenuItem();
  JMenu jMenuHelp = new JMenu();
  JMenuItem jMenuHelpAbout = new JMenuItem();
  JToolBar jToolBar = new JToolBar();
  JButton jButtonOpenUncrypted = new JButton();
  JButton jButtonSave = new JButton();
  JButton jButtonHelp = new JButton();
  JLabel statusBar = new JLabel();
  BorderLayout borderLayout1 = new BorderLayout();
  JTabbedPane stegoTabbedPane = new JTabbedPane();
  JButton jButtonOpenCrypt = new JButton();

  String algos[][] = {
                      {"LSB", "trex.Algo.StegoAlgoLSB", ""},
                      {"LSB with pass phrase", "trex.Algo.StegoAlgoLSBpp", ""},
                      {"LSB in frequency domain (Fourier transformed)", "trex.Algo.StegoAlgoFFT", ""},
//                      {"LSB with config dialog", "trex.Algo.StegoAlgoLSBcfg", ""},
//                      {"Invalid","trex.StegData", null},
                      {"Custom (unloaded)", "", null}};

  ButtonGroup menuAlgoMenuGroup;
  JRadioButtonMenuItem algoItems[];
  JRadioButtonMenuItem currentSelectedMenuAlgo = null;
  Class currentAlgoClass = null;

  /**
   * Picture-, data and config panel.
   * These are inserted in the tabbed pane.
   */
  Component uncryptPanel;
  DataPanel dataPanel;
  JPanel passPanel;
  Component cryptPanel;
  Component combinedPanel;
  Component histoPanel;

  /** This instance's data */
  StegData myData;
  JMenu jMenuExtra = new JMenu();
  JMenuItem jMenuExtraGC = new JMenuItem();
  JMenu jMenuAlgo = new JMenu();
  JMenuItem jMenuAlgoLoad = new JMenuItem();
  JMenuItem jMenuFileSaveCrypted = new JMenuItem();
  JMenuItem jMenuFileLoadCrypted = new JMenuItem();
  JMenuItem jMenuFileLoadUncrypted = new JMenuItem();

  /**
   * Reconstruct the panels containg the pictures (cryted, uncrypted, combined).
   */
  public void resetPicPanels() {
    // Construct panels (text is fallback).
    uncryptPanel = TRexUtil.makePanel(myData.getUncrypted(), "No uncrypted image available.");
    cryptPanel = TRexUtil.makePanel(myData.getCrypted(), "No encrypted image available.");
    combinedPanel = new CombinedPanel(myData.getUncrypted(),
                                      myData.getCrypted(),
                                      this);

    histoPanel = new HistoPanel(this,
                                myData.getUncrypted(),
                                myData.getCrypted(),
                                Color.blue,
                                Color.yellow,
                                "Blue: uncrypted image - " +
                                "Yellow: crypted image - " +
                                "White: both",
                                200,
                                HistoPanel.CHANNEL_VALUE);

    // reset the new panels in the tabbed pane.
    stegoTabbedPane.setComponentAt(0, uncryptPanel);
    stegoTabbedPane.setComponentAt(3, cryptPanel);
    stegoTabbedPane.setComponentAt(4, combinedPanel);
    stegoTabbedPane.setComponentAt(5, histoPanel);
  }

  /**
   * Reset the data panel.
   */
  public void resetDataPanel() {
    dataPanel = new DataPanel(this, myData);
    stegoTabbedPane.setComponentAt(1, dataPanel);
  }

  /** Construct the frame. */
  public MainFrame(String picfilename,
                   String datafilename,
                   String algorithmname) {
    enableEvents(AWTEvent.WINDOW_EVENT_MASK);
    try {
      jbInit();
    }
    catch(Exception e) {
      e.printStackTrace();
    }

    initDefltAlgo();

    // create new data object
    if (myData == null) myData = new StegData(this, currentAlgoClass);

    dataPanel = new DataPanel(this, myData);

    stegoTabbedPane.addTab("Uncrypted Image", uncryptPanel);
    stegoTabbedPane.addTab("Data", dataPanel);
    stegoTabbedPane.addTab("Pass phrase/Config", passPanel);
    stegoTabbedPane.addTab("Crypted Image", cryptPanel);
    stegoTabbedPane.addTab("Combined View", combinedPanel);
    stegoTabbedPane.addTab("Histograms", histoPanel);
    stegoTabbedPane.setSelectedIndex(0);

    if (picfilename != null) {
      boolean success = myData.openUncrypted(picfilename);
      if (!success)
        JOptionPane.showMessageDialog(this,
                        "File could not be read",
                        "Error",
                        JOptionPane.ERROR_MESSAGE);
    }

    if (datafilename != null) {
      File f = new File(datafilename);
      dataPanel.loadData(f);
    }

    if (algorithmname != null) {
      loadAlgoClass(algorithmname);
    }

    // is this clean? pic panels are not set up yet, but inserted into tabbed pane...
    resetPicPanels();
    resetConfigPassPane();

  }

  /**
   * Setup up the config dialog or pass phrase dialog (depending on current
   * algorithm).
   */
  private void resetConfigPassPane() {
    if ((myData != null) && (myData.hasConfigDialog())) {
      passPanel = new JPanel(new BorderLayout());
      JButton updBut = new JButton("Restart algorithm with this setup!");
      passPanel.add(myData.getConfigDialog(), BorderLayout.CENTER);
      passPanel.add(updBut, BorderLayout.SOUTH);

      updBut.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(ActionEvent e) {
          if (myData != null) myData.paramsChanged();
          resetPicPanels();
          resetDataPanel();
        }
      });

    } else if ((myData != null) && (myData.hasPassPhrase())) {
      passPanel = new PassPhrasePanel(myData, "");
    } else {
      passPanel = null;
    }

    if (passPanel == null)
      passPanel = TRexUtil.makeTextPanel("No pass phrase or config available.");

    stegoTabbedPane.setComponentAt(2, passPanel);
  }

  /**
   * Event handler for algorithm selections in the menu.
   */
  private void algoSelected(ActionEvent e) {
    Class algoClass = null;
    boolean failure = false;

    try {
      algoClass = ClassLoader.getSystemClassLoader().
        loadClass(e.getActionCommand());

    }

    catch (Exception ex) {
      JOptionPane.showMessageDialog(this,
                      "The selected Algorithm could not be loaded!",
                      "Error",
                      JOptionPane.ERROR_MESSAGE);
      failure = true;
    }

    if (!failure && (!trex.Algo.StegoAlgo.class.isAssignableFrom(algoClass))) {
      failure = true;
      JOptionPane.showMessageDialog(this,
                      "The selected class is not a valid algorithm class.",
                      "Error",
                      JOptionPane.ERROR_MESSAGE);
      failure = true;
    }

    if (!failure) {
      failure = !myData.setAlgo(algoClass);
    }

    if (failure) {
      if (currentSelectedMenuAlgo != null)
        currentSelectedMenuAlgo.setSelected(true);
    }

    if (!failure) {
      myData.paramsChanged();

      currentSelectedMenuAlgo = (JRadioButtonMenuItem)e.getSource();
      resetPicPanels();
      resetDataPanel();
      resetConfigPassPane();
    }
  }

  /**
   * Initialize default Algo class
   */
  private void initDefltAlgo() {
    try {
      currentAlgoClass = ClassLoader.getSystemClassLoader().loadClass(algos[0][1]);
    }

    catch (Exception ex) {
      JOptionPane.showMessageDialog(this,
                      "The default algorithm could not be loaded!\nExiting...",
                      "Error",
                      JOptionPane.ERROR_MESSAGE);
      System.exit(-1);
    }
  }

  /**
   * Initialize the "Algorithm" menu, inserting the known algorithms
   * and installing a handler.
   */
  private void initAlgoMenu() {
    if (algoItems != null) {
      for (int i = 0; i < algoItems.length; i++) {
        if (algoItems[i] != null) jMenuAlgo.remove(algoItems[i]);
      }
    }

    algoItems = new JRadioButtonMenuItem[algos.length];
    menuAlgoMenuGroup = new ButtonGroup();
    for (int i = 0; i < algos.length; i++) {
      if ((algos[i][0] != null) && (algos[i][1] != null)) {
        boolean sel = (i == 0) ? true : false;
        JRadioButtonMenuItem jMenuAlgoAny = new JRadioButtonMenuItem(algos[i][0], sel);
        jMenuAlgoAny.setActionCommand(algos[i][1]);
        if (algos[i][2] == null)
          jMenuAlgoAny.setEnabled(false);

        if (sel) currentSelectedMenuAlgo = jMenuAlgoAny;
        algoItems[i] = jMenuAlgoAny;

        jMenuAlgoAny.addActionListener(new ActionListener() {
          public void actionPerformed(ActionEvent e) {
            algoSelected(e);
          }
        });

        jMenuAlgo.add(jMenuAlgoAny);
        menuAlgoMenuGroup.add(jMenuAlgoAny);
      }
    }
  }

  /**Component initialization*/
  private void jbInit() throws Exception  {
    //setIconImage(Toolkit.getDefaultToolkit().createImage(MainFrame.class.getResource("[Your Icon]")));
    contentPane = (JPanel) this.getContentPane();
    contentPane.setLayout(borderLayout1);
    this.setSize(new Dimension(700, 500));
    this.setTitle("TRex");
    statusBar.setText(" ");
    jMenuFile.setMnemonic('F');
    jMenuFile.setText("File");
    jMenuFileExit.setMnemonic('x');
    jMenuFileExit.setText("Exit");
    jMenuFileExit.setAccelerator(javax.swing.KeyStroke.getKeyStroke(81, java.awt.event.KeyEvent.CTRL_MASK, false));
    jMenuFileExit.addActionListener(new ActionListener()  {
      public void actionPerformed(ActionEvent e) {
        jMenuFileExit_actionPerformed(e);
      }
    });
    jMenuHelp.setMnemonic('H');
    jMenuHelp.setText("Help");
    jMenuHelpAbout.setMnemonic('A');
    jMenuHelpAbout.setText("About");
    jMenuHelpAbout.addActionListener(new ActionListener()  {
      public void actionPerformed(ActionEvent e) {
        jMenuHelpAbout_actionPerformed(e);
      }
    });
    jButtonOpenUncrypted.setIcon(new ImageIcon(MainFrame.class.getResource("openFile.gif")));
    jButtonOpenUncrypted.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jButtonOpenUncrypted_actionPerformed(e);
      }
    });
    jButtonOpenUncrypted.setToolTipText("Open Uncrypted");
    jButtonSave.setIcon(new ImageIcon(MainFrame.class.getResource("closeFile.gif")));
    jButtonSave.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jButtonSave_actionPerformed(e);
      }
    });
    jButtonSave.setToolTipText("Save Crypted");
    jButtonHelp.setIcon(new ImageIcon(MainFrame.class.getResource("help.gif")));
    jButtonHelp.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jButtonHelp_actionPerformed(e);
      }
    });
    jButtonHelp.setToolTipText("Help");
    jButtonOpenCrypt.setToolTipText("Open Crypted");
    jButtonOpenCrypt.setIcon(new ImageIcon(MainFrame.class.getResource("openCrypt.gif")));
    jButtonOpenCrypt.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jButtonOpenCrypt_actionPerformed(e);
      }
    });
    jMenuExtra.setMnemonic('E');
    jMenuExtra.setText("Extra");
    jMenuExtraGC.setMnemonic('G');
    jMenuExtraGC.setText("Start Garbage Collection");
    jMenuExtraGC.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jMenuExtraGC_actionPerformed(e);
      }
    });
    jMenuAlgo.setMnemonic('A');
    jMenuAlgo.setText("Algorithm");
    jMenuAlgoLoad.setMnemonic('L');
    jMenuAlgoLoad.setText("Load custom algorithm");
    jMenuAlgoLoad.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jMenuAlgoLoad_actionPerformed(e);
      }
    });
    jMenuFileLoadUncrypted.setMnemonic('L');
    jMenuFileLoadUncrypted.setText("Load Uncrypted...");
    jMenuFileLoadUncrypted.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jMenuFileLoadUncrypted_actionPerformed(e);
      }
    });
    jMenuFileLoadCrypted.setToolTipText("");
    jMenuFileLoadCrypted.setMnemonic('C');
    jMenuFileLoadCrypted.setText("Load Crypted...");
    jMenuFileLoadCrypted.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jMenuFileLoadCrypted_actionPerformed(e);
      }
    });
    jMenuFileSaveCrypted.setMnemonic('S');
    jMenuFileSaveCrypted.setText("Save Crypted...");
    jMenuFileSaveCrypted.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jMenuFileSaveCrypted_actionPerformed(e);
      }
    });
    jToolBar.add(jButtonOpenUncrypted);
    jToolBar.add(jButtonOpenCrypt, null);
    jToolBar.add(jButtonSave);
    jToolBar.add(jButtonHelp);
    jMenuFile.add(jMenuFileLoadUncrypted);
    jMenuFile.add(jMenuFileLoadCrypted);
    jMenuFile.add(jMenuFileSaveCrypted);
    jMenuFile.addSeparator();
    jMenuFile.add(jMenuFileExit);
    jMenuHelp.add(jMenuHelpAbout);
    jMenuBar1.add(jMenuFile);
    jMenuBar1.add(jMenuAlgo);
    jMenuBar1.add(jMenuExtra);
    jMenuBar1.add(jMenuHelp);
    this.setJMenuBar(jMenuBar1);
    contentPane.add(jToolBar, BorderLayout.NORTH);
    contentPane.add(statusBar, BorderLayout.SOUTH);
    contentPane.add(stegoTabbedPane, BorderLayout.CENTER);
    jMenuExtra.add(jMenuExtraGC);
    jMenuAlgo.add(jMenuAlgoLoad);
    jMenuAlgo.addSeparator();

    initAlgoMenu();
  }
  /** File | Exit action performed. */
  protected void jMenuFileExit_actionPerformed(ActionEvent e) {
    System.exit(0);
  }
  /** Help | About action performed. */
  protected void jMenuHelpAbout_actionPerformed(ActionEvent e) {
    MainFrame_AboutBox dlg = new MainFrame_AboutBox(this);
    Dimension dlgSize = dlg.getPreferredSize();
    Dimension frmSize = getSize();
    Point loc = getLocation();
    dlg.setLocation((frmSize.width - dlgSize.width) / 2 + loc.x, (frmSize.height - dlgSize.height) / 2 + loc.y);
    dlg.setModal(true);
    dlg.show();
  }
  /** Overridden so we can exit when window is closed. */
  protected void processWindowEvent(WindowEvent e) {
    super.processWindowEvent(e);
    if (e.getID() == WindowEvent.WINDOW_CLOSING) {
      jMenuFileExit_actionPerformed(null);
    }
  }
  /** Event handler for "About" box button. */
  void jButtonHelp_actionPerformed(ActionEvent e) {
    jMenuHelpAbout_actionPerformed(null);
  }

  /**
   * Open the file chooser depending on passed dialog type.
   * @param Title Window title.
   * @param excheck Check for readability before closing. Useful for "Open" dialogs.
   */
  String fileDialog(final String Title, boolean excheck) {
    // create chooser
    JFileChooser fc = new JFileChooser();
    fc.addChoosableFileFilter(new ImageFilter());
    fc.setFileView(new ImageFileView());
    fc.setAccessory(new ImagePreview(fc));

    // open or save title?
    int returnVal = fc.showDialog(MainFrame.this, Title);

    // if "OK" was pressed, open/save file
    if (returnVal != JFileChooser.APPROVE_OPTION)
      return null; // Dialog was not closed by OK Button

    if (excheck && !fc.getSelectedFile().canRead()) {
       JOptionPane.showMessageDialog(this,
                       "File could not be read",
                       "Error",
                       JOptionPane.ERROR_MESSAGE);
       return null;
    }

    return fc.getSelectedFile().toString();
  }

  /**
   * Event handler for "Open Uncrypted" button.
   * Delegates "open" to myData object.
   */
  void jButtonOpenUncrypted_actionPerformed(ActionEvent e) {
    String filename;
    boolean success;
    if ((filename = fileDialog("Open uncrypted...", true)) != null) {
      success = myData.openUncrypted(filename);
      if (!success)
        JOptionPane.showMessageDialog(this,
                        "File could not be read",
                        "Error",
                        JOptionPane.ERROR_MESSAGE);
      else
        resetPicPanels();
    }
  }

  /**
   * Event handler for "Open Crypted" button.
   * Delegates "open" to myData object.
   */
  void jButtonOpenCrypt_actionPerformed(ActionEvent e) {
    String filename;
    boolean success;
    if ((filename = fileDialog("Open crypted...", true)) != null) {
      success = myData.openCrypted(filename);
      if (!success) {
        JOptionPane.showMessageDialog(this,
                        "File could not be read",
                        "Error",
                        JOptionPane.ERROR_MESSAGE);
      } else {
        resetPicPanels();
        resetDataPanel();
      }
    }
  }

  /**
   * Event handler for "Save Crypted" button.
   * Delegates "save" to myData object.
   */
  void jButtonSave_actionPerformed(ActionEvent e) {
    if (myData.getCrypted() == null) {
      JOptionPane.showMessageDialog(this,
                                    "Crypted Image currently unset.\nNothing to save.",
                                    "Error",
                                    JOptionPane.ERROR_MESSAGE);
    } else {
      String filename;
      boolean success;
      if ((filename = fileDialog("Save...", false)) != null) {
        success = myData.saveCrypted(filename);
        if (!success)
          JOptionPane.showMessageDialog(this,
                          "File could not be written.",
                          "Error",
                          JOptionPane.ERROR_MESSAGE);
      }
    }
  }

  /** Extra | Garbage Collection action performed. */
  protected void jMenuExtraGC_actionPerformed(ActionEvent e) {
    System.gc();
  }

  protected void loadAlgoClass(String className) {
    Class newClass;
    String info;

    try {
      newClass = ClassLoader.getSystemClassLoader().
        loadClass(className);
    }

    catch (Exception ex) {
      JOptionPane.showMessageDialog(this,
                      "The selected Algorithm could not be loaded!",
                      "Error",
                      JOptionPane.ERROR_MESSAGE);
      return;
    }

    if (!(trex.Algo.StegoAlgo.class.isAssignableFrom(newClass))) {
      JOptionPane.showMessageDialog(this,
                      "The selected class is not a valid algorithm class.",
                      "Error",
                      JOptionPane.ERROR_MESSAGE);
      return;
    }

    try {
      StegoAlgo instance = (StegoAlgo)(newClass.newInstance());
      info = instance.getInfo();
    }
    catch (Exception ex) {
      JOptionPane.showMessageDialog(this,
                      "An unknown Error occured loading the class.",
                      "Error",
                      JOptionPane.ERROR_MESSAGE);
      return;
    }

    algoItems[algos.length-1].setActionCommand(className);
    algoItems[algos.length-1].setText(info);
    algoItems[algos.length-1].setEnabled(true);

    algoItems[algos.length-1].setSelected(true);
    algoSelected(new ActionEvent(algoItems[algos.length-1], 1, className));
  }

  /** Algorithm | Load Algorithm action performed. */
  protected void jMenuAlgoLoad_actionPerformed(ActionEvent e) {
    String className = JOptionPane.showInputDialog(
                          "Enter a class name to load:");

    loadAlgoClass(className);
  }

  /**
   * File | Load uncrypted action performed.
   * Simply pass to button event handler.
   */
  protected void jMenuFileLoadUncrypted_actionPerformed(ActionEvent e) {
    jButtonOpenUncrypted_actionPerformed(e);
  }

  /**
   * File | Load crypted action performed.
   * Simply pass to button event handler.
   */
  protected void jMenuFileLoadCrypted_actionPerformed(ActionEvent e) {
    jButtonOpenCrypt_actionPerformed(e);
  }

  /**
   * File | Save crypted action performed.
   * Simply pass to button event handler.
   */
  protected void jMenuFileSaveCrypted_actionPerformed(ActionEvent e) {
    jButtonSave_actionPerformed(e);
  }
}