/*
 * $Id: benchmark.c 931 2007-03-26 12:55:34Z bastian $
 *
 * Benchmarking module for OpenSER
 *
 * Copyright (C) 2007 Collax GmbH
 *                    (Bastian Friedrich <bastian.friedrich@collax.com>)
 *
 * This file is part of openser, a free SIP server.
 *
 * openser is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version
 *
 * openser is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <sys/time.h>
#include <stdlib.h>

#include "../../sr_module.h"
#include "../../mi/mi.h"

#include "benchmark.h"
#include "fixup_str2int.h"

MODULE_VERSION

/*
 * Module destroy function prototype
 */
static void destroy(void);

/*
 * Module child-init function prototype
 */
static int child_init(int rank);

/*
 * Module initialization function prototype
 */
static int mod_init(void);


/*
 * Exported parameters
 */
int enable_global = 0;
int granularity = 100;
int loglevel = L_INFO;


/*
 * Internal timers
 */

struct benchmark_timer_t timers[BENCHMARK_TIMERS];


/*
 * Reload perl interpreter - reload perl script. Forward declaration.
 */
struct mi_root* bm_mi_enable_global(struct mi_root *cmd_tree, void *param);
struct mi_root* bm_mi_granularity(struct mi_root *cmd_tree, void *param);
struct mi_root* bm_mi_loglevel(struct mi_root *cmd_tree, void *param);
struct mi_root* bm_mi_enable_timer(struct mi_root *cmd_tree, void *param);



/*
 * Exported functions
 */
static cmd_export_t cmds[] = {
	{ "start_timer", usr_start_timer, 1, fixup_str2int, REQUEST_ROUTE
							   | FAILURE_ROUTE
							   | ONREPLY_ROUTE
							   | BRANCH_ROUTE },
	{ "log_timer", usr_log_timer, 1, fixup_str2int, REQUEST_ROUTE | FAILURE_ROUTE
							   | ONREPLY_ROUTE | BRANCH_ROUTE },

	{"load_bm", (cmd_function)load_bm, 0,  0, 0},
	{ 0, 0, 0, 0, 0 }
};


/*
 * Exported parameters
 */
static param_export_t params[] = {
	{"enable", INT_PARAM, &enable_global},
	{"granularity", INT_PARAM, &granularity},
	{"loglevel", INT_PARAM, &loglevel},
	{ 0, 0, 0 }
};


/*
 * Exported MI functions
 */
static mi_export_t mi_cmds[] = {
	/* { "bm_enable_global", bm_mi_enable_global, MI_NO_INPUT_FLAG,  0,  0  },
	{ "bm_enable_timer",  bm_mi_enable_timer, MI_NO_INPUT_FLAG,  0,  0  },
	{ "bm_granularity",  bm_mi_granularity, MI_NO_INPUT_FLAG,  0,  0  },
	{ "bm_loglevel", bm_mi_loglevel, MI_NO_INPUT_FLAG, 0, 0 }, */
	{ 0, 0, 0, 0}
};


/*
 * Module interface
 */
struct module_exports exports = {
	"benchmark", 
	DEFAULT_DLFLAGS,
	cmds,       /* Exported functions */
	params,     /* Exported parameters */
	0,          /* exported statistics */
	mi_cmds,    /* exported MI functions */
	0,          /* exported pseudo-variables */
	mod_init,   /* module initialization function */
	0,          /* response function */
	destroy,    /* destroy function */
	child_init  /* child initialization function */
};


/****************/


static int child_init(int rank)
{
	return 0;
}


/*
 * Reinit through fifo.
 */
struct mi_root* perl_mi_reload(struct mi_root *cmd_tree, void *param)
{
	if (1) {
		return init_mi_tree( 200, MI_OK_S, MI_OK_LEN);
	} else {
		return init_mi_tree( 500, "Perl reload failed", 18);
	}

}

void reset_timer(int i) {
	timers[i].calls = 0;
	timers[i].sum = 0;
	timers[i].last_max = 0;
	timers[i].last_min = LONG_MAX;
	timers[i].last_sum = 0;
	timers[i].global_max = 0;
	timers[i].global_min = LONG_MAX;
}

void reset_timers() {

	int i;

	for (i = 0; i < BENCHMARK_TIMERS; i++) {
		reset_timer(i);
	}
}

/*
 * mod_init
 * Called by openser at init time
 */
static int mod_init() {

	int ret = 0;
	int i;

	LOG(L_INFO, "benchmark: initializing\n");

	reset_timers();

	for (i = 0; i < BENCHMARK_TIMERS; i++) {
		timers[i].enabled = 0;
	}

	return ret;
}

/*
 * destroy
 * called by openser at exit time
 */
static void destroy(void)
{
}

/*
 * timer_active().
 * Global enable mode can be:
 * -1 - All timing disabled
 *  0 - Timing enabled, watch for single timers enabled (default: off)
 *  1 - Timing enabled for all timers
 */

inline int timer_active(int i) {
	if ((enable_global > 0) || ((enable_global == 0) && (timers[i].enabled > 0)))
		return 1;
	else
		return 0;
}

inline long usecdiff(struct timeval *t1, struct timeval *t2) {
	long secdiff;
	long usecdiff;

	secdiff = t2->tv_sec - t1->tv_sec;
	usecdiff = t2->tv_usec - t1->tv_usec;

	if (secdiff < 0) {
		secdiff = -secdiff;
		usecdiff = -usecdiff;
	}

	while (usecdiff < 0) {
		usecdiff += 1000000; /* next second */
		secdiff -=1;
	}

	return secdiff*1000000 + usecdiff;
}


/*
 * start_timer()
 */

int start_timer(int id) {
	if ((id >= BENCHMARK_TIMERS) || (id < 0)) {
		LOG(L_ERR, "benchmark: Timer id %d not available. Please use timers 0 through %d\n",
				id,
				BENCHMARK_TIMERS-1);
		return -1;
	}

	if (timer_active(id))
		gettimeofday(&(timers[id].start), NULL);

	return 1;
}

int usr_start_timer(struct sip_msg* _msg, char* timer, char *foobar) {
	int i = (int)timer;

	return start_timer(i);
}


/*
 * log_timer()
 */

int log_timer(int id) {
	struct timeval now;
	long diff;

	if ((id >= BENCHMARK_TIMERS) || (id < 0)) {
		LOG(L_ERR, "benchmark: Timer id %d not available. Please use timers 0 through %d\n",
				id,
				BENCHMARK_TIMERS-1);
		return -1;
	}

	if (!timer_active(id))
		return 1;

	gettimeofday(&now, NULL);
	diff = usecdiff(&(timers[id].start), &now);

	/* What to do
	 * - update min, max, sum
	 * - if granularity hit: Log, reset min/max
	 */

	timers[id].sum += diff;
	timers[id].last_sum += diff;
	timers[id].calls++;
	
	if (diff < timers[id].last_min)
		timers[id].last_min = diff;

	if (diff > timers[id].last_max)
		timers[id].last_max = diff;

	if (diff < timers[id].global_min)
		timers[id].global_min = diff;

	if (diff > timers[id].global_max)
		timers[id].global_max = diff;


	if ((timers[id].calls % granularity) == 0) {
		LOG(loglevel, "benchmark (timer %d): reporting msgs/total/min/max/avg - since last report: %i/%li/%li/%li/%f | global: %li/%li/%li/%li/%f\n",
				id,
				granularity,
				timers[id].last_sum,
				timers[id].last_min,
				timers[id].last_max,
				((double)timers[id].last_sum)/granularity,
				timers[id].calls,
				timers[id].sum,
				timers[id].global_min,
				timers[id].global_max,
				((double)timers[id].sum)/timers[id].calls);

		timers[id].last_sum = 0;
		timers[id].last_max = 0;
		timers[id].last_min = LONG_MAX;
	}

	return 1;
}	

int usr_log_timer(struct sip_msg* _msg, char* timer, char* mystr) {

	int i = (int)timer;

	return log_timer(i);
}

/* API Binding */

int load_bm( struct bm_binds *bmb)
{
	if(bmb==NULL)
		return -1;

	bmb->bm_start	 = start_timer;
	bmb->bm_log	 = log_timer;

	return 1;
}


/* End of file */
